<?php

namespace App\Http\Controllers\Admin;
use App\Models\Investment;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class InvestmentController extends Controller
{
    
    
    public function showInvestments()
    {
        return view('admin.investment',['investments'=> Investment::get(),]);
    }


    public function createInvestments(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'minimum_amount' => 'required|numeric|min:0',
            'maximum_amount' => 'required|numeric|gt:minimum_amount',
            'interest' => 'required|numeric|min:0|max:100',
            'setting_file' => 'required|image|mimes:jpeg,png,jpg|max:2048', // Only images, max 2MB
            'description' => 'nullable|string|max:1000',
        ]);

        try {
            $filePath = $request->file('setting_file')->store('settings', 'public');

            $cryptoInvestment = Investment::create([
                'name' => $validated['name'],
                'minimum_amount' => $validated['minimum_amount'],
                'maximum_amount' => $validated['maximum_amount'],
                'interest' => $validated['interest'],
                'setting_file_path' => $filePath,
                'description' => $validated['description'] ?? '',
            ]);


            return redirect()->route('admin.show_investments')
                ->with('success', 'Crypto investment created successfully.');

        } catch (\Exception $e) {


           // dd($e);
            return redirect()->back()
                ->with('error', 'Error saving crypto investment: ' . $e->getMessage())
                ->withInput();
        }
    }


    public function destroyInvestments($id)
    {
        $investment = Investment::find($id);

    if (!$investment) {
        return redirect()->back()->with('error', 'Investment not found.');
    }

    // Optional: Delete the associated image file
    if ($investment->setting_file_path) {
        Storage::disk('public')->delete($investment->setting_file_path);
    }

    $investment->delete();

    return redirect()->route('admin.show_investments')
         ->with('success', 'Investment deleted successfully!');
    }




}
